<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Company;
use App\Models\Emailtemplate;
use Socialite;
use Illuminate\Support\Facades\DB;
use App\Http\Helpers\UploadHelper;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    /**
     * Create user
     *
     * @param  [string] name
     * @param  [string] email
     * @param  [string] password
     * @param  [string] password_confirmation
     * @return [string] message
     */
    public function signup(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'email' => 'required|string|email|unique:users',
            'password' => 'required|string|confirmed'
        ]);
        $user = new User([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password)
        ]);
        $user->save();
        return response()->json([
            'message' => 'Successfully created user!'
        ], 201);
    }

    /**
     * Login user and create token
     *
     * @param  [string] email
     * @param  [string] password
     * @param  [boolean] remember_me
     * @return [string] access_token
     * @return [string] token_type
     * @return [string] expires_at
     */
    public function login(Request $request)
    {

        $request->validate([
            'email' => 'required|string|email',
            'password' => 'required|string',
            'remember_me' => 'boolean'
        ]);
        $credentials = request(['email', 'password']);
        if (!Auth::guard('web')->attempt($credentials, $request->remember_me)) {
            return $request->all();
            $failed_user = User::where('email', $credentials['email'])->first();
            if ($failed_user) {
                $failed_user->attempt = ($failed_user->attempt + 1);
                $failed_user->save();
            }
            return response()->json([
                'status' => false,
                'message' => 'Invalid Credentials'
            ], 200);
        }

        $user = User::where('email', $request->email)->first();
        $authentication = $this->authenticateLogin($user);
        if (!$authentication['status']) {
            return response()->json($authentication, 200);
        }
        $tokenResult = $user->createToken('Personal Access Token');
        $token = $tokenResult->token;
        $user->attempt = 0;
        $user->save();
        if ($request->remember_me)
            $token->expires_at = Carbon::now()->addWeeks(1);
        $token->save();
        return response()->json([
            'access_token' => $tokenResult->accessToken,
            'token_type' => 'Bearer',
            'expires_at' => Carbon::parse(
                $tokenResult->token->expires_at
            )->toDateTimeString(),
            'status' => true,
            'message' => 'Login Successfull'
        ]);
    }

    /**
     * Login Authentication
     * @param [User] $user, [Int] $google_auth
     * @return [Array]
     */
    protected function authenticateLogin(User $user, $google_auth = 0)
    {
        if ($user->active == 0 || $user->attempt >= 3) {
            return [
                'status' => false,
                'message' => 'Your account is temporarily locked. Please ask your company admin to unlock it'
            ];
        }
        if (($google_auth == 0) && is_null($user->email_verified_at) && !empty($user->email_verification_token)) {
            return [
                'status' => false,
                'message' => 'Please Verify your Email first'
            ];
        }
        return [
            'status' => true,
        ];
    }

    /**
     * Logout user (Revoke the token)
     *
     * @return [string] message
     */
    public function logout(Request $request)
    {
        $request->user()->token()->revoke();
        return response()->json([
            'message' => 'Successfully logged out'
        ]);
    }

    /**
     * Get the authenticated User
     *
     * @return [json] user object
     */
    public function user(Request $request)
    {
        return response()->json($request->user());
    }

    /**
     * Get the authenticated User
     *
     * @return [json] user object
     */
    public function profile(Request $request)
    {
        $user = request()->user();
        $permissions = $user->getAllPermissions();
        $user->all_permissions = $permissions->pluck('name')->toArray();
        $user->church_ids = DB::table('church_user')->where('user_id', request()->user()->id)->get()->pluck('church_id');
        $user->rgs = DB::table('requester_groups')->whereIn('company_id', $user->church_ids)->get()->pluck('group_id');
        return response()->json($user);
    }


    /**
     * @param: Request
     * @return: array
     */
    public function forgotPassword(Request $request)
    {
        $user = User::where('email', $request->email)->first();
        if ($user) {
            $id_hash = base64_encode($user->id);
            $getChurch = DB::table('church_user')->where('user_id', $user->id)->first();
            $church_id = (isset($getChurch->church_id) && $getChurch->church_id > 0) ? $getChurch->church_id : 0;
            $emailtemplate = Emailtemplate::where('name', 'forget_password')->where('church_id', $church_id)->first();

            $email_template_id = $emailtemplate ? $emailtemplate->id : 0;
            $body = $emailtemplate ? $emailtemplate->mail_body : '';
            $subject = $emailtemplate ? $emailtemplate->mail_subject : '';
            $url = url('/set-password') . '?hash=' . $id_hash;
            $body = str_replace("{{set_password_link}}", $url, $body);
            $mail = new \App\Http\Helpers\SendEmail;
            try {
                $mail->send($user->email, env('MAIL_FROM_ID'), $body, $subject, $email_template_id, null);
                return [
                    'status' => true,
                    'message' => 'An Email is sent to your registered email'
                ];
            } catch (\Exception $e) {
                return [
                    'status' => false,
                    'message' => $e->getMessage()
                ];
            }
        } else {
            return [
                'status' => false,
                'message' => 'Sorry! This Email is not registered with us'
            ];
        }
    }


    /**
     * @param: Request
     * @return: array
     */
    public function updateProfile(Request $request)
    {
        $user = request()->user();
        // if(User::where('email', $request->email)->where('id','<>', $user->id)->count() > 0){
        //     return [
        //         'status' => false,
        //         'message' => 'This Email is already assigned to another User\'s Account'
        //     ];
        // }
        // $user->email = $request->email;
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->phone = str_contains($request->phone, '+1') ? $request->phone :  '+1' . $request->phone;
        $user->notification = ($request->notification) ? 1 : 0;
        if (!empty($request->password))
            $user->password = bcrypt($request->password);

        if ($request->hasFile('avatar')) {
            $file = $request->file('avatar');
            try {
                $user->avatar = UploadHelper::UploadFile($file, 'user');
            } catch (\Exception $e) {
                return [
                    'status' => false,
                    'message' => $e->getMessage()
                ];
            }
        }
        $user->save();
        return [
            'status' => true,
            'message' => 'Profile Updated Successfully'
        ];
    }

    public function changePassword(Request $request)
    {
        $user = $request->user();
        if (Hash::check($request->old_password, $user->password)) {
            $user->password = bcrypt($request->new_password);
            $user->save();
            return [
                'status' => true,
                'message' => 'Password has been changed successfully'
            ];
        } else {
            return [
                'status' => false,
                'message' => 'Old password did not match'
            ];
        }
    }


    public function gauth(Request $request)
    {
        $g_user = $request->g_user;
        $target = [];
        $access_token = '';
        $count = 0;
        foreach ($g_user as $key => $value) {
            if ($count == 1) {
                $target = $value;
                break;
            }
            $count++;
        }
        $access_token = (!empty($target) && !empty($target['access_token'])) ? $target['access_token'] : '';
        if (!empty($target['access_token'])) {

            $social_user = Socialite::driver('google')->userFromToken($access_token);
            $email = $social_user->getEmail();
            $avatar = $social_user->getAvatar();
            $user = User::where('email', $email)->first();
            if ($user) {
                $user->google_token = $access_token;
                $authentication = $this->authenticateLogin($user, 1);
                if (!$authentication['status']) {
                    return response()->json($authentication, 200);
                }
                if (is_null($user->avatar))
                    $user->avatar = $avatar;
                if (is_null($user->email_verified_at)) {
                    $user->email_verified_at = Carbon::now();
                    $user->email_verification_token = null;
                    $user->active = 1;
                }
                $user->save();
                return [
                    'status' => true,
                    'token' => $user->createToken('Personal Access Token')->accessToken
                ];
            } else {
                return [
                    'status' => false,
                    'message' => 'Sorry! This Email is not registered with us'
                ];
            }
        }
        return [
            'status' => false,
            'message' => 'Sorry! Token is invalid'
        ];
    }

    public function setPassword(Request $request)
    {
        $id = base64_decode($request->hash);
        $user = User::find($id);
        $user->password = bcrypt($request->password);
        $user->save();
        return [
            'status' => true,
            'message' => 'Password has been changed. Please log in'
        ];
    }

    /**
     * @param: Request
     * @return: Collection
     */
    public function companies(Request $request)
    {
        return Company::select('id', 'name as text')
            ->where('name', 'like', "%$request->search%")
            ->get();
    }

    /**
     * @param: Request
     * @return: Collection
     */
    public function fetchUsers(Request $request)
    {
        return User::select(DB::raw("CONCAT(COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as value"), 'id as index')
            ->where('company_id', $request->company_id)
            ->where('id', '!=', $request->user_id)
            ->whereNotNull('email_verified_at')
            ->get();
    }


    public function impersonate(Request $request)
    {
        $user = User::find($request->user_id);
        if ($user) {
            $tokenResult = $user->createToken('Personal Access Token');
            $token = $tokenResult->token;
            $token->save();
            return response()->json([
                'access_token' => $tokenResult->accessToken,
                'token_type' => 'Bearer',
                'expires_at' => Carbon::parse(
                    $tokenResult->token->expires_at
                )->toDateTimeString()
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Invalid User ID'
            ], 200);
        }
    }
}
