<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class Analytic extends BaseModel {

    use SoftDeletes;

    protected static $logAttributes = ['id', 'wallet_id', 'product_id', 'channel_id', 'country_id', 'date', 'total_searches', 'monetized_searches', 'coverage', 'ad_impressions', 'ad_clicks', 'monetized_ctr', 'publisher_cpc', 'amount'];
    protected $fillable = ['id', 'wallet_id', 'product_id', 'channel_id', 'country_id', 'date', 'total_searches', 'monetized_searches', 'coverage', 'ad_impressions', 'ad_clicks', 'monetized_ctr', 'publisher_cpc', 'amount'];
    public $default_sort = 'id';
    public $default_sort_order = 'desc';
    public $model = 'analytics';
    public $select_fields = ['analytics.id', 'analytics.company_id', 'analytics.wallet_id', 'wallets.wallet as wallet', 'analytics.product_id', 'products.product as product', 'analytics.channel_id', 'channels.channel as channel', 'analytics.country_id', 'countries.name as country', 'analytics.date', 'analytics.total_searches', 'analytics.monetized_searches', 'analytics.coverage', 'analytics.ad_impressions', 'analytics.ad_clicks', 'analytics.monetized_ctr', 'analytics.publisher_cpc', 'analytics.amount'];
    public $fields_array = array(
        'id' => array(
            'field_name' => 'id',
            'db_name' => 'id',
            'type' => 'text',
            'placeholder' => 'Id',
            'listing' => true,
            'show_in_form' => false,
            'sort' => true,
            'default_sort' => true,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'date' => array(
            'field_name' => 'date',
            'db_name' => 'date',
            'type' => 'text',
            'placeholder' => 'Date',
            'listing' => true,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'product_id' => array(
            'field_name' => 'product',
            'db_name' => 'product_id',
            'type' => 'autocomplete',
            'multiple' => false,
            'related_table' => 'products',
            'searchable' => 'product',
            'placeholder' => 'Product',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50',
            'options' => []
        ),
        'wallet_id' => array(
            'field_name' => 'wallet',
            'db_name' => 'wallet_id',
            'type' => 'autocomplete',
            'multiple' => false,
            'related_table' => 'wallets',
            'searchable' => 'wallet',
            'placeholder' => 'Wallet',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50',
            'options' => []
        ),
        'country_id' => array(
            'field_name' => 'country',
            'db_name' => 'country_id',
            'type' => 'autocomplete',
            'multiple' => false,
            'related_table' => 'countries',
            'searchable' => 'name',
            'placeholder' => 'Country',
            'listing' => true,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50',
            'options' => []
        ),
        'total_searches' => array(
            'field_name' => 'channel',
            'db_name' => 'channel_id',
            'type' => 'text',
            'placeholder' => 'Total Searches',
            'listing' => true,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'coverage' => array(
            'field_name' => 'coverage',
            'db_name' => 'coverage',
            'placeholder' => 'Coverage',
            'type' => 'text',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'total_monetized_searches' => array(
            'field_name' => 'total_monetized_searches',
            'db_name' => 'total_monetized_searches',
            'type' => 'text',
            'placeholder' => 'Total Monetized Searches',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'ad_impressions' => array(
            'field_name' => 'ad_impressions',
            'db_name' => 'ad_impressions',
            'type' => 'text',
            'placeholder' => 'Ad Impressions',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'ad_clicks' => array(
            'field_name' => 'ad_clicks',
            'db_name' => 'ad_clicks',
            'type' => 'text',
            'placeholder' => 'Ad Clicks',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'monetized_ctr' => array(
            'field_name' => 'monetized_ctr',
            'db_name' => 'monetized_ctr',
            'type' => 'text',
            'placeholder' => 'Monetized CTR',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'publisher_cpc' => array(
            'field_name' => 'publisher_cpc',
            'db_name' => 'publisher_cpc',
            'type' => 'text',
            'placeholder' => 'Publisher CPC',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'amount' => array(
            'field_name' => 'amount',
            'db_name' => 'amount',
            'type' => 'text',
            'placeholder' => 'Amount',
            'listing' => false,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        )
    );
    
    public function getList(Request $request) {
        $show = $request->show ? $request->show : 2;
        $query = $this->getListQuery($request);
        $results = $query->paginate($show);
        $results_count = $query->count();
        return array('results_count' => $results_count, 'results' => $results, 'widget' => array(
                'revenue' => $this->getRevenue($request),
                'ad_clicks' => $this->getAdClicks($request),
                'change' => $this->getChangePercent($request),
                'eom_projection' => $this->getEomProjection($request)
        ));
    }
    
    public function getListQuery(Request $request){
        $query = parent::getListQuery($request);
        $query->leftJoin('wallets','analytics.wallet_id','wallets.id');
        $query->leftJoin('countries','analytics.country_id','countries.id');
        $query->leftJoin('channels','analytics.channel_id','channels.id');
        $query->leftJoin('products','analytics.product_id','products.id');
        
        if($request->user()->can("list-all-analytics")){
            $query = $query;
        } else if($request->user()->can("list-group-analytics")){
            $query->where('analytics.company_id', '=', $request->user()->company_id);
        }
        
        if(isset($request->company) && !is_null($request->company)){
            $query->where('analytics.company_id',$request->company);
        }
        if(isset($request->month) && !is_null($request->month)){
            $term = $request->month.'/'.$request->year;
            $query->where('analytics.date', 'LIKE', "$request->month%$request->year");
        }
        return $query;
    }

    public function getCompanies() {
        $company_arr = array();
        $companies = \App\Models\Company::All();
        array_push($company_arr, array('index' => 'select', 'value' => 'Select Company'));
        foreach ($companies as $company) {
            array_push($company_arr, array('index' => $company->id, 'value' => $company->name));
        }
        return $company_arr;
    }

    public function getPublishers() {
        $pubs_arr = array();
        $pubs = \App\Models\Publisher::All();
        array_push($pubs_arr, array('index' => 'select', 'value' => 'Select Publisher'));
        foreach ($pubs as $pub) {
            array_push($pubs_arr, array('index' => $pub->id, 'value' => $pub->name));
        }
        return $pubs_arr;
    }
    
    public function getRevenue(Request $request){
        $wquery = DB::table('analytics')->select(DB::raw("SUM(`amount`) as sum_amount"));
        if($request->user()->can("list-all-analytics")){
            $wquery = $wquery;
        } else if($request->user()->can("list-group-analytics")){
            $wquery->where('analytics.company_id', '=', $request->user()->company_id);
        }
        if(isset($request->company) && !is_null($request->company)){
            $wquery->where('company_id',$request->company);
        }
        if(isset($request->month) && !is_null($request->month)){
            $term = $request->month.'/'.$request->year;
            $wquery->where('date', 'LIKE', "$request->month%$request->year");
        }
        $response = $wquery->first();
        if(!empty($response)){
            return is_null($response->sum_amount) ? 0 : round($response->sum_amount,2);
        }
        return 0;
    }
    
    public function getAdClicks(Request $request){
        $wquery = DB::table('analytics')->select(DB::raw("SUM(`ad_clicks`) as sum_ad_clicks"));
        if($request->user()->can("list-all-analytics")){
            $wquery = $wquery;
        } else if($request->user()->can("list-group-analytics")){
            $wquery->where('analytics.company_id', '=', $request->user()->company_id);
        }
        if(isset($request->company) && !is_null($request->company)){
            $wquery->where('company_id',$request->company);
        }
        if(isset($request->month) && !is_null($request->month)){
            $term = $request->month.'/'.$request->year;
            $wquery->where('date', 'LIKE', "$request->month%$request->year");
        }
        $response = $wquery->first();
        if(!empty($response)){
            return is_null($response->sum_ad_clicks) ? 0 : $response->sum_ad_clicks;
        }
        return 0;
    }
    
    public function getEomProjection(Request $request){
        $month = $request->month ? $request->month : date('m');
        $year = $request->year ? $request->year : date('Y');
        $eom_projection = 0;
        $wquery = DB::table('analytics')->select(DB::raw("avg(`amount`) as avg_amount"));
        if($request->user()->can("list-all-analytics")){
            $wquery = $wquery;
        } else if($request->user()->can("list-group-analytics")){
            $wquery->where('analytics.company_id', '=', $request->user()->company_id);
        }
        if(isset($request->company) && !is_null($request->company)){
            $wquery->where('analytics.company_id',$request->company);
        }
        $avg = $wquery->where('analytics.date', 'LIKE', "$month%$year")->first();
        $d=cal_days_in_month(CAL_GREGORIAN,$month,$year);
        $eom_projection = $avg->avg_amount * $d;
        return round($eom_projection,2);
    }
    
    public function getChangePercent(Request $request){
        $request->month = $request->month ? $request->month : date('m');
        $request->year = $request->year ? $request->year : date('Y');
        $ticker = 'up';
        $val = 0;
        $wquery = DB::table('analytics')->select(DB::raw("avg(`amount`) as avg_amount"));
        if($request->user()->can("list-all-analytics")){
            $wquery = $wquery;
        } else if($request->user()->can("list-group-analytics")){
            $wquery->where('analytics.company_id', '=', $request->user()->company_id);
        }
        if(isset($request->company) && !is_null($request->company)){
            $wquery->where('analytics.company_id',$request->company);
        }
        if(isset($request->month) && !is_null($request->month)){
            $term = $request->month.'/'.$request->year;
            $result = $wquery->where('analytics.date', 'LIKE', "$request->month%$request->year")->first();
        }
        
        $cur_dt = $request->year.'-'.$request->month.'-31';
        $datestring = "$cur_dt first day of last month";
        $dt = date_create($datestring);
        $new_term = $dt->format('m/Y');
        $last_month = $dt->format('n');
        $last_year = $dt->format('Y');
        
        $wquery2 = DB::table('analytics')->select(DB::raw("avg(`amount`) as avg_amount"));
        if($request->user()->can("list-all-analytics")){
            $wquery2 = $wquery2;
        } else if($request->user()->can("list-group-analytics")){
            $wquery2->where('analytics.company_id', '=', $request->user()->company_id);
        }
        if(isset($request->company) && !is_null($request->company)){
            $wquery2->where('analytics.company_id',$request->company);
        }
        //return $result->toSql();   
        $last_month_result = $wquery2->where('analytics.date', 'LIKE', "$last_month%$last_year")->first();
        
        if($result && $last_month_result){
            if($last_month_result > $result){
                $ticker = 'down';
            }
            if($result->avg_amount && $last_month_result->avg_amount){
                $val = abs(round($result->avg_amount - $last_month_result->avg_amount,2));
            }
        }
        
        return array('ticker'=>$ticker, 'value' => $val);
    }
    
    public function getWidgetQuery(Request $request, $wquery){
        if(isset($request->company) && !is_null($request->company)){
            $wquery->where('analytics.company_id',$request->company);
        }
        if(isset($request->month) && !is_null($request->month)){
            $term = $request->month.'/'.$request->year;
            $wquery->where('analytics.date', 'LIKE', "$request->month%$request->year");
        }
        return $wquery;
    }

}
