<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Excel;
use App\Models\Company;

class Import extends BaseModel {

    use SoftDeletes;

    protected static $logAttributes = ['filename', 'original_filename', 'attempted', 'rows', 'success', 'failure', 'reverted', 'company_id'];
    protected $fillable = ['id', 'filename', 'original_filename', 'attempted', 'rows', 'success', 'failure', 'reverted', 'company_id'];
    public $searchable = ['imports.id', 'imports.filename', 'imports.original_filename'/* , 'companies.name' */];
    public $default_sort = 'id';
    public $default_sort_order = 'asc';
    public $model = 'import';
    public $select_fields = ['imports.id', 'imports.filename', 'imports.original_filename', 'imports.attempted', 'imports.rows', 'imports.success', 'companies.name as company', 'companies.id as company_id', 'imports.updated_at as imported_on', 'imports.failure', 'imports.reverted', 'imports.error_filename'];
    //public $join_fields = array('left' => array('comments.id = post_id'));
    public $fields_array = array(
        'id' => array(
            'field_name' => 'id',
            'db_name' => 'id',
            'type' => 'text',
            'placeholder' => 'Id',
            'listing' => true,
            'sort' => true,
            'default_sort' => true,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'filename' => array(
            'field_name' => 'filename',
            'db_name' => 'filename',
            'type' => 'file',
            'placeholder' => 'Filename',
            'listing' => true,
            'sort' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'rows' => array(
            'field_name' => 'success',
            'db_name' => 'success',
            'type' => 'text',
            'placeholder' => 'Rows',
            'listing' => true,
            'sort' => true,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50',
            'show_in_field' => false
        ),
        'company_id' => array(
            'field_name' => 'company_id',
            'db_name' => 'company_id',
            'type' => 'select',
            'placeholder' => 'Company',
            'listing' => false,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'company' => array(
            'field_name' => 'company',
            'db_name' => 'company_id',
            'type' => 'select',
            'placeholder' => 'Company',
            'listing' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50',
            'show_in_form' => false
        ),
        'error_filename' => array(
            'field_name' => 'error_filename',
            'db_name' => 'error_filename',
            'type' => 'button',
            'placeholder' => 'Error Filename',
            'listing' => false,
            'sort' => false,
            'default_sort' => false,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'imported_on' => array(
            'field_name' => 'imported_on',
            'db_name' => 'updated_at',
            'type' => 'date',
            'placeholder' => 'Imported On',
            'listing' => true,
            'default_sort' => false,
            'required' => true,
            'value' => '',
            'width' => '50',
            'show_in_form' => false
        )
    );

    function __construct() {
        parent::__construct();
        $this->fields_array['company_id']['options'] = $this->getCompanies();
    }

    // Accessor Function
    public function getErrorFilenameAttribute($value) {
        if (is_null($value))
            return '';
        else {
            $company = $this->company()->first();
            if ($company) {
                $folder_name = $company->name . '_' . $company->id;
                return asset('error_csv/' . $value);
            } else {
                return '';
            }
        }
    }

    public function getListQuery(Request $request) {
        $sort = $request->sort ? $request->sort : $this->getModel()->default_sort;
        $sort_order = $request->sort_order ? $request->sort_order : $this->getModel()->default_sort_order;
        $trash = isset($request->trash) ? $request->trash : 'false';
        $table = $this->getModel()->getTable();
        $raw_sql = "$table.*";
        if (!empty($this->select_fields)) {
            $raw_sql = implode(',', $this->select_fields);
        }
        $query = Import::select(DB::raw($raw_sql))
                ->leftJoin('companies', 'imports.company_id', '=', 'companies.id');

        if ($request->user()->can('list-all-imports')) {
            $query = $query;
            //dd('All');
        } else if ($request->user()->can('list-group-imports')) {
            $query->where('imports.company_id', $request->user()->company_id);
            //dd('grp');
        }

        // Filter records using where clause
        $where_clause = $request->where_clause ? $request->where_clause : '';
        if ($where_clause != '') {
            $where_clause = json_decode($where_clause, 1);
            $where_fields = $where_clause['where_fields'];
            $where_values = $where_clause['where_values'];

            foreach ($where_fields as $key => $where_field) {
                if (is_array($where_values[$key]))
                    $query->whereIn($where_field, $where_values[$key]);
                else
                    $query->where($where_field, $where_values[$key]);
            }
        }

        if (!empty($this->searchable) && $request->search != '') {
            $concat_fields = 'concat(';
            foreach ($this->searchable as $field) {
                $concat_fields .= 'COALESCE(' . $field . ", ''),'',";
            }
            $concat_fields = rtrim($concat_fields, ",'',");
            $concat_fields .= ')';
            $query->where([[DB::raw($concat_fields), 'like', "%$request->search%"]]);
        }
//        $query->where('imports.deleted_at', null);
        if ($trash == 'false') {
            $query->whereNull($table . '.deleted_at');
        } else {
            $query->whereNotNull($table . '.deleted_at');
        }
        $query->orderBy($sort, $sort_order);

        return $query;
    }

    public $relationship = array(
        'comment' => array(
            'type' => 'one-to-many',
            'lhs_key' => 'id', // db field name
            'rhs_key' => 'id', // db field name
            'lhs_table' => '', // optional (if table name is different fron model name)
            'rhs_table' => '', // optional (if table name is different fron model name)
            'pivot' => array(
                'table' => '', // name of the pivot table
                'lhs_key' => 'id', // db field name
                'rhs_key' => 'id', // db field name
            ) // optional (if many to many then required)
        )
    );
    public $relationship_types = array('one-to-one', 'one-to-one-inverse', 'one-to-many', 'one-to-many-inverse', 'many-to-many');

    public function publish(Request $request) {
        if (!$request->hasFile('file')) {
            return array('status' => false, 'message' => 'Please use a supported file');
        } else if (!$request->has('company_id')) {
            return array('status' => false, 'message' => 'Please select company');
        }
        $company = Company::find($request->company_id);
        if (!$company) {
            return array('status' => false, 'message' => 'Selected company does not exist');
        }
        $file = $request->file('file');
        $file_extension = $file->getClientOriginalExtension();
        if ($file_extension !== 'csv' && $file_extension !== 'xlsx' && $file_extension !== 'xls') {
            return array('status' => false, 'message' => 'Please use a supported file format');
        }
        $current_time = strtotime(Carbon::now()->toDateTimeString());
        $mime_type = $file->getMimeType();
        $file_size = $file->getSize();
        $org_file_name = $file->getClientOriginalName();
        $file_name = $company->name . '_' . $current_time . '_' . $file->getClientOriginalName();

        $new_import = new Import();
        $new_import->filename = $file_name;
        $new_import->original_filename = $org_file_name;
        $new_import->company_id = $request->company_id;
        $new_import->save();

        if ($new_import) {
            Storage::disk('uploads')->putFileAs(
                    'imports/', $file, $file_name
            );
            return array('status' => true, 'message' => 'File uploaded successfully');
        }
    }

    public function company() {
        return $this->belongsTo('App\Models\Company');
    }

}
